set more off 
pause off
set logtype text
set mem 500M

tempfile bushee_out
tempfile currtemp

*************** DESCRIPTION *************************************
* Computes percentage ownership by each type of institution
* in each firm
*
* Inputs: 	2.intermediate\bushee_detailed --> detailed ownership data
*			CRSP --> CRSP datapull to map CUSIP to GVKEY
*			LinkingTable --> CRSP-Compustat linking table available in WRDS
*
* Outputs: bushee_firmmap --> firm-level share of ownership, by type
*
* Note:
*	- rdate is the date for which the holdings are valid; BUT
*	- fdate is the date for which share price and shares outstanding are given in TR
* We therefore compute total share ownership by rdate, and pull share 
* price and shares outstanding from CRSP using ncusip8 for the given rdate
*****************************************************************

* aggregate by class
use 2.intermediate\bushee_detailed, clear
drop mgrno_mapped
order cusip year invclass shares 
keep cusip year invclass shares 

collapse (sum) shares , by (cusip year invclass )

keep cusip year invclass shares 
reshape wide shares, i(cusip year) j(invclass ["QIX" "TRA" "DED" "NA"])
rename cusip cusip8

compress
save `bushee_out', replace


**** Grab price and shares outstanding from CRSP for given dates 

* Prepare crsp dataset
use 0.raw_inputs\crsp_data, clear
g cusip8 = substr(ncusip, 1, 8)
g year = year(date)
keep if month(date)== 12
g mktcap = abs(prc) * shrout

* Some ncusips are not populated. We could replace them with the most recent cusip 
* including the following code, but it would not properly replicate history 
* replace cusip8 = substr(cusip,1,8) if cusip8 == "" 
drop if cusip8 == "" 
save `currtemp', replace

/* ---------------------------------- */
/* Get CRSP permco, shrout and price  */
/* ---------------------------------- */
use `bushee_out', clear
merge 1:1 cusip8 year using `currtemp', keepusing(permco permno mktcap shrout)
drop if _merge ~= 3
drop _merge


* In case of multiple cusip8 (i.e., multiple permno) for a given permco, 
* select cusip8 with largest market capitalization
g negmktcap = -mktcap
sort permco year negmktcap
bys permco year : keep if _n==1
save `currtemp', replace

/* ------------*/
/* Grab Gvkey  */
/* ------------*/

* join bushee dataset with CCM table to obtain GVKEY for chosen permno
use 0.raw_inputs\linkingTable, clear
rename lpermco permco
joinby permco using `currtemp', unmatched(none)

* Keep only if gvkey is relevant for chosen years
keep if year(linkdt) <= year & year(linkenddt) >= year

* drop duplicates
sort gvkey year negmktcap
bys gvkey year : keep if _n==1
drop negmktcap

/* --------------------------*/
/*  Compute shares and save  */
/* --------------------------*/

* Compute pct of share ownership.
egen sharesinstowned = rowtotal(shares*)

foreach X in QIX DED TRA NA{
g pctsharetot`X' = shares`X'/(1000*shrout)
g pctshareins`X' = shares`X'/sharesinstowned

replace pctsharetot`X' = 1 if pctsharetot`X' > 1 & pctsharetot`X' ~= .
replace pctshareins`X' = 1 if pctshareins`X' > 1 & pctshareins`X' ~= .

replace pctsharetot`X' = 0 if pctsharetot`X' == . & year <= 2013
replace pctshareins`X' = 0 if pctshareins`X' == . & year <= 2013

replace pctsharetot`X' = . if year == 1980
replace pctshareins`X' = . if year == 1980
}

g pctinsown = sharesinstowned/(1000*shrout)
replace pctinsown = 1 if pctinsown > 1 & pctinsown ~= .
replace pctinsown = 0 if pctinsown == . & year <= 2013

keep gvkey year cusip8 permco pct* 
destring, replace

save 2.intermediate\bushee_firmmap, replace

use 2.intermediate\bushee_firmmap,clear
* Selected test for manual replication: 
g test1 =  (pctsharetotQIX - 0.0089701) if cusip8 == "75120310" & year == 1989
g test2 = (pctshareinsQIX -.2724388) if cusip8 == "75120310" & year == 1989
su test*
drop test*
pause



